##################################################################
# Description:   User administration                             #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2000 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   23.07.2000   Initial coding                          #
##################################################################

package modules::user;

use NoK::tools;

$VERSION = "1.00";
$NAME = "User";
$MODULE = "modules::user";
$DELIMITER = ":";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_events'        => 'join part quit nick msg userident',
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_tools'         => new NoK::tools,
         '_modulehandler' => $modulehandler,
         '_data'          => {},
         '_channels'      => {},
         '_identify'      => {},
         '_filename'      => 'user'
   };

   if (defined($config)) {
      if (defined($config->get('USERFILE'))) {
         $self->{'_filename'} = $config->get('USERFILE');
      }
   }

   bless $self, $proto;
   $self->load();
   return $self;
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   if (length($self->{'_channels'}->{$nick}) == 0) {
      $self->{'_channels'}->{$nick} = $channel;
   } else {
      $self->{'_channels'}->{$nick} = $self->{'_channels'}->{$nick}."\t".$channel;
   }
   if ($self->checkmode($nick,"o") == 1) {
      $self->{'_connection'}->mode($channel, "+o", $nick);
   }
}

sub on_part {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   my @chan = split("\t",$self->{'_channels'}->{$nick});
   $self->{'_channels'}->{$nick} = "";
   foreach $schannel (@chan) {
      if ($schannel ne $channel) {
         if (length($self->{'_channels'}->{$nick}) == 0) {
            $self->{'_channels'}->{$nick} = $channel;
         } else {
            $self->{'_channels'}->{$nick} = $self->{'_channels'}->{$nick}."\t".$channel;
         }
      }
   }
   if (length($self->{'_channels'}->{$nick}) == 0) {
      $self->{'_identify'}->{$nick} = undef;
      $self->{'_channels'}->{$nick} = undef;
   }
}

sub on_quit {
   $self = shift;
   my ($nick, $userhost) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   $self->{'_identify'}->{$nick} = undef;
   $self->{'_channels'}->{$nick} = undef;
}

sub on_nick {
   $self = shift;
   my ($nick, $userhost, $newnick) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   $newnick = $self->{'_tools'}->realnick($newnick);
   if (length($self->{'_identify'}->{$nick}) > 0) {
      $self->{'_identify'}->{$newnick} = $self->{'_identify'}->{$nick};
   }
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   if (substr($text,0,length($self->{'_config'}->get('COMMANDPREFIX'))) eq $self->{'_config'}->get('COMMANDPREFIX')) {
      my @words = split(/ /,substr($text,length($self->{'_config'}->get('COMMANDPREFIX'))));
      my $cmd = uc(shift(@words));
      if ($cmd eq "USERREG") { $self->register($nick,@words);}
      if ($cmd eq "USERIDENT") { $self->identify($nick,@words); }
      if ($cmd eq "USERPWD") { $self->changepwd($nick,@words); }
      if ($cmd eq "USERMODE") { $self->changemode($nick,@words); }
      if ($cmd eq "HELP") { $self->help($nick,@words); }
   }
}

sub on_userident {
   $self = shift;
   my ($nick,$user) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   if ($self->checkmode($nick,"o") == 1) {
      foreach my $channel (split(/;/,$self->{'_config'}->get('IRCCHANNEL'))) {
         $self->{'_connection'}->mode($channel, "+o", $nick);
      }
   }
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub load {
   $self = shift;
   if (-r $self->{'_filename'}) {
      open FD, $self->{'_filename'} or die "Seen: Error reading '".$self->{'_filename'}."'.\n";
      while (<FD>) {
         chomp;
         my ($nick,$pwd,$mode) = split($DELIMITER);
         my $data = $pwd. $DELIMITER.$mode;
         $self->{'_data'}->{$nick} = $data;
      }
      close FD;
   }
}

sub save {
   $self = shift;
   open FD, ">".$self->{'_filename'} or die "Seen: Error writing '".$self->{'_filename'}."'.\n";
   *DATA = $self->{'_data'};
   foreach $nick (keys %DATA) {
      print FD $nick.$DELIMITER.$DATA{$nick}."\n";
   }
   close FD;
}

sub error {
   $self = shift;
   my ($nick,$cmd,$text) = @_;
   $self->{'_connection'}->privmsg($nick,"Error: ".$cmd." - ".$text);
}

sub register {
   $self = shift;
   my $nick = shift;

   # REGISTER [Nick] Password
   my $user = $nick;
   if ((scalar(@_) < 1) || (scalar(@_) > 2)) {
      $self->error($nick,"REGISTER","Not correct amount of arguments.");
      return;
   }
   my $pwd = shift;
   if (scalar(@_) > 0) {
      $user = $pwd;
      $pwd = shift;
   }
   $user = lc($user);

   if (length($self->{'_data'}->{$user}) > 0) {
      $self->error($nick,"REGISTER","'$user' already registered.");
      return;
   }
   my $salt = join('', ('.', '/', 0..9, 'A'..'Z','a'..'z')[rand 64, rand 64]);
   $pwd = crypt($pwd,$salt);
   $self->{'_data'}->{$user} = $pwd;
   $self->save;
   $self->{'_connection'}->privmsg($nick,"'$user' registered.");
}

sub identify {
   $self = shift;
   my $nick = shift;

   # IDENTIFY [Nick] Password
   my $user = $nick;
   if ((scalar(@_) < 1) || (scalar(@_) > 2)) {
      $self->error($nick,"IDENTIFY","Not correct amount of arguments.");
      return;
   }
   my $pwd = shift;
   if (scalar(@_) > 0) {
      $user = $pwd;
      $pwd = shift;
   }
   $user = lc($user);
   if (length($self->{'_data'}->{$user}) > 0) {
      my ($spwd,$smode) = split($DELIMITER,$self->{'_data'}->{$user});
      if ($spwd eq crypt($pwd,substr($spwd,0,2))) {
         $self->{'_identify'}->{$nick} = $user;
         $self->{'_connection'}->privmsg($nick,"You are identified as '$user'.");
         if (defined($self->{'_eventhandler'})) {
            $self->{'_eventhandler'}->action("userident",$nick,$user);
         }
      } else {
         $self->error($nick,"IDENTIFY","Wrong password.");
      }
   } else {
      $self->error($nick,"IDENTIFY","Unknown user '$user'.");
   }
}

sub checkmode {
   $self = shift;
   my $nick = shift;
   my $mode = shift;
   my $user = $nick;
   my $pwd = undef;
   if (scalar(@_) > 0) {
      $pwd = shift;
   }
   if (scalar(@_) > 0) {
      $user = $pwd;
      $pwd = shift;
   }
   if (defined($pwd)) {
      $self->identify($nick, $user, $pwd);
   }
   if (length($self->{'_identify'}->{$nick}) > 0) {
      $user = $self->{'_identify'}->{$nick};
      my ($spwd, $smode) = split($DELIMITER,$self->{'_data'}->{$user});
      if (index($smode,$mode) == -1) {
         return 0;
      } else {
         return 1;
      }
   } else {
      return 0;
   }
}

sub changepwd {
   $self = shift;
   my $nick = shift;
   my $newpwd = shift;
   my $user = $nick;
   if (scalar(@_) > 0) {
      $user = $newpwd;
      $newpwd = shift;
   }
   if (length($self->{'_identify'}->{$nick}) > 0) {
      $user = lc($user);
      if (($user ne lc($nick)) && ($user ne $self->{'_identify'}->{$nick})) {
         # change foreign passwords only for master
         if ($self->checkmode($nick,"m") == 0) {
            $self->error($nick,"PASSWORD","You cannot change foreign passwords.");
            return;
         }
      } else {
         $user = $self->{'_identify'}->{$nick};
      }
      my $salt = join('', ('.', '/', 0..9, 'A'..'Z','a'..'z')[rand 64, rand 64]);
      my ($spwd, $smode) = split($DELIMITER,$self->{'_data'}->{$user});
      $newpwd = crypt($newpwd,$salt);
      $self->{'_data'}->{$user} = $newpwd.$DELIMITER.$smode;
      $self->{'_connection'}->privmsg($nick,"Password for '$user' changed.");
      $self->save();
   } else {
      $self->error($nick,"PASSWORD","'$user' not identified.");
   }
}

sub changemode {
   $self = shift;
   my $nick = shift;
   my $mode = shift;
   my $user = $nick;
   if (scalar(@_) > 0) {
      $user = $mode;
      $mode = shift;
   }
   if (length($self->{'_identify'}->{$nick}) > 0) {
      if ($self->checkmode($nick,"m") == 0) {
         $self->error($nick,"MODE","You cannot change modes.");
         return;
      }
      $user = lc($user);
      my ($spwd, $smode) = split($DELIMITER,$self->{'_data'}->{$user});
      my $sign = substr($mode,0,1);
      if ($sign eq "+") {
         my $cmode = substr($mode,1,1);
         if (index($smode,$cmode) == -1) {
            $smode = $smode . $cmode;
            $self->{'_data'}->{$user} = $spwd.$DELIMITER.$smode;
            $self->save();
            $self->{'_connection'}->privmsg($nick,"Mode '".substr($mode,1,1)."' added to '$user'");
         }
      }
      if ($sign eq "-") {
         $_ = $smode;
         my $cmode = substr($mode,1,1);
         s/$cmode//;
         $smode = $_;
         $self->{'_data'}->{$user} = $spwd.$DELIMITER.$smode;
         $self->save();
         $self->{'_connection'}->privmsg($nick,"Mode $cmode removed from '$user'");
      }
      if ($sign eq "?") {
         $self->{'_connection'}->privmsg($nick,"'$user' has mode '$smode'");
      }
   } else {
      $self->error($nick,"MODE","'$user' not identified.");
   }
}

sub help {
   $self = shift;
   my $nick = shift;
   if (scalar(@_) > 0) {
      $cmd = lc(shift);
      if ($cmd eq "userreg") {
         $self->{'_connection'}->privmsg($nick,"userreg [<user>] <password> - Register a new account with a password. If you don't want to use your nickname enter <user>.");
         $self->{'_connection'}->privmsg($nick,"Example: userreg foo bar - Opens a new acount as user 'foo' with the password 'bar'.");
      }
      if ($cmd eq "userident") {
      }
   } else {
      # print list
      $self->{'_connection'}->privmsg($nick,"userreg         - Register/Open new account.");
      $self->{'_connection'}->privmsg($nick,"userident       - Identify/Login.");
      $self->{'_connection'}->privmsg($nick,"userpwd         - Change password.");
      if ($self->checkmode($nick,"m") == 1) {
         $self->{'_connection'}->privmsg($nick,"usermode        - Change mode.");
      }
   }
}

sub exists {
   $self = shift;
   my $user = lc(shift);
   if (length($self->{'_data'}->{$user}) > 0) {
      return 1;
   } else {
      return 0;
   }
}

sub get_user {
   $self = shift;
   my $nick = shift;
   if (length($self->{'_identify'}->{$nick}) > 0) {
      return $self->{'_identify'}->{$nick};
   } else {
      return "";
   }
}
