##################################################################
# Description:   Logging idle time on IRC                        #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2001 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   29.12.2001   Initial coding                          #
##################################################################

package modules::idle;

$VERSION = "1.00";
$NAME = "Idle";
$MODULE = "modules::idle";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_userlist'      => {},
         '_topduration'   => {},
         '_topstart'      => {},
         '_events'        => 'join part quit nick msg public names action mode kick notice',
         '_delimiter'     => "\t"
   };
   bless $self, $proto;

   $self->top_load();

   return $self;
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $self->add($channel,$nick);
}

sub on_part {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $self->delete($channel,$nick);
}

sub on_quit {
   $self = shift;
   my ($nick, $userhost, $reason) = @_;
   $self->delete("*",$nick);
}

sub on_nick {
   $self = shift;
   my ($nick, $userhost, $newnick) = @_;
   $self->rename($nick,$newnick);
   $self->reset("*",$newnick);
}

sub on_public {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;
   my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."IDLETOP");
   my $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      $self->top_ranking($nick);
   } else {
      $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."IDLE");
      $search = substr(uc($text),0,length($cmd));
      if ($search eq $cmd) {
         $self->ranking_current($nick);
      }
   }
   $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."HELP");
   $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      $_ = substr($text,length($cmd));
      s/^\s+//;
      s/\s+$//;
      $self->help($nick,$_);
   }
   $self->reset($channel,$nick);
}

sub on_action {
   $self = shift;
   my ($channel, $nick, $userhost, $action) = @_;
   $self->reset($channel,$nick);
}

sub on_mode {
   $self = shift;
   my ($channel, $nick, $userhost, $mode, $arg) = @_;
   $self->reset($channel,$nick);
}

sub on_kick {
   $self = shift;
   my ($channel, $nick, $userhost, $user, $reason) = @_;
   $self->delete($channel,$nick);
}

sub on_names {
   $self = shift;
   my ($channel, $list) = @_;
   foreach $nick (split(/ /,$list)) {
      if (length($nick)>0) { $self->add($channel,$nick); }
   }
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   if ($nick eq $self->{'_connection'}->nick()) { return; }

   my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."IDLETOP");
   my $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      my @word = split(/ /,$text);
      if (scalar(@word) > 1) {
         $self->top_nick($nick,$word[1]);
      } else {
         $self->top_ranking($nick);
      }
   } else {
      $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."IDLE");
      $search = substr(uc($text),0,length($cmd));
      if ($search eq $cmd) {
         $self->ranking_current($nick);
      }
   }
}

sub on_notice {
   $self = shift;
   my ($nick, $userhost, $eargs) = @_;
   $self->reset("*",$nick);
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub ranking_current {
   $self = shift;
   $nick = shift;
   $curtime = time();
   *DATA = $self->{'_userlist'};
   my $count = 1;
   foreach $chanuser (sort { $DATA{$a} <=> $DATA{$b} } keys %DATA) {
      my $idle = $self->format_time($curtime - $DATA{$chanuser});
      my ($chan,$user) = split($self->{'_delimiter'},$chanuser);
      $self->{'_connection'}->privmsg($nick, $count.". ".$user." on ".$chan." (".$idle.")");
      if ($count >= $self->{'_config'}->get('IDLELISTSIZE')) { return; }
      $count++;
   }
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub reset {
   $self = shift;
   my ($channel, $nick) = @_;

   # clean nick
   $nick = $self->clean_nick($nick);

   my $mynick = "";
   if (defined($self->{'_config'})) {
      $mynick = $self->{'_connection'}->nick();
   }
   if ($nick ne $mynick) {
      my $tcurrent = time();
      if ($channel eq "*") {
         *DATA = $self->{'_userlist'};
         foreach $chanuser (keys %DATA) {
            my ($chan,$user) = split($self->{'_delimiter'},$chanuser);
            if ($user eq $nick) {
               if ($self->top_check($nick,$self->{'_userlist'}->{$chan.$self->{'_delimiter'}.$nick},$tcurrent) > 0) {
                  $self->top_save();
               }
               $self->{'_userlist'}->{$chan.$self->{'_delimiter'}.$nick} = $tcurrent;
            }
         }
      } else {
         my $key = $channel.$self->{'_delimiter'}.$nick;
         if (defined($self->{'_userlist'}->{$key})) {
            if ($self->top_check($nick,$self->{'_userlist'}->{$key},$tcurrent) > 0) {
               $self->top_save();
            }
            $self->{'_userlist'}->{$key} = $tcurrent;
         }
      }
   }
}

sub add {
   $self = shift;
   my ($channel, $nick) = @_;

   # clean nick
   $nick = $self->clean_nick($nick);

   # ignore nick ?
   my $ignorelist = "";
   if (defined($self->{'_config'})) {
      $ignorelist = lc(";".$self->{'_config'}->get('IDLEIGNORELIST').";");
      if (index($ignorelist,lc(";".$nick.";")) != -1) { return; }
   }

   if (($nick ne $mynick) && ($channel ne "*")) {
      $self->{'_userlist'}->{$channel.$self->{'_delimiter'}.$nick} = time();
   }
}

sub clean_nick {
   $self = shift;
   $nick = shift; 
   $_ = $nick;
   if (/@(.*)/) {
      $_ = $1;
   } elsif (/\+(.*)/) {
      $_ = $1;  
   }
   return $_;
}

sub format_time {
   $self = shift;
   $sec = shift;
   $hour = int($sec / 3600);
   $sec = $sec - ($hour * 3600);
   $min = int($sec / 60);
   $sec = $sec - ($min * 60);
   $text = $hour.":".substr("0".$min,-2).":".substr("0".$sec,-2);
   return $text;
}

sub rename {
   $self = shift;
   my ($oldnick, $newnick) = @_;
   $oldnick = $self->clean_nick($oldnick); 
   $newnick = $self->clean_nick($newnick); 
   *DATA = $self->{'_userlist'};
   foreach $chanuser (keys %DATA) {
      my ($chan,$user) = split($self->{'_delimiter'},$chanuser);
      if ($user eq $oldnick) {
          my $value = $DATA{$chanuser};
          $self->{'_userlist'}->{$chan.$self->{'_delimiter'}.$newnick} = $DATA{$chanuser};
          delete($DATA{$chanuser});
      }
   }
}

sub delete {
   $self = shift;
   my ($channel, $nick) = @_;
   $nick = $self->clean_nick($nick);
   my $ignorelist = "";
   if (defined($self->{'_config'})) {
      $ignorelist = ";".$self->{'_config'}->get('IDLEIGNORELIST').";";
   }
   if ($channel eq "*") {
      *DATA = $self->{'_userlist'};
      foreach $chanuser (keys %DATA) {
         my ($chan,$user) = split($self->{'_delimiter'},$chanuser);
         if ($user eq $nick) {
            delete($DATA{$chanuser});
         }
      }
   } else {
      my $key = $channel.$self->{'_delimiter'}.$nick;
      if (defined($self->{'_userlist'}->{$key})) {
         delete($self->{'_userlist'}->{$key});
      }
   }
}

sub top_load {
   $self = shift;
   if (defined($self->{'_config'})) {
      my $filename = $self->{'_config'}->get('IDLETOPLIST');
      if (-r $filename) {
         open FD, $filename or die "Idle: Error reading '".$filename."'.\n";
         while (<FD>) {
            chomp;
            my ($nick,$duration,$start) = split( $self->{'_delimiter'});
            $self->{'_topduration'}->{$nick} = $duration;
            $self->{'_topstart'}->{$nick} = $start;
        }
        close FD;
      }
   }
}

sub top_save {
   $self = shift;
   if (defined($self->{'_config'})) {
      my $filename = $self->{'_config'}->get('IDLETOPLIST');
      open FD, ">".$filename;
      *DATA = $self->{'_topduration'};
         foreach my $key (keys %DATA) {
         print FD $key.$self->{'_delimiter'}.$DATA{$key}.$self->{'_delimiter'}.$self->{'_topstart'}->{$key}."\n";
      }
      close FD;
   }
}

sub top_check {
   $self = shift;
   my ($nick, $t_start, $t_current) = @_;
   my $t_duration = $t_current - $t_start;
   my $change = 1;
   if (defined($self->{'_topduration'}->{$nick})) {
      if ($t_duration <= $self->{'_topduration'}->{$nick}) {
         $change = 0;
      }
   }

   if ($change > 0) {
      $self->{'_topduration'}->{$nick} = $t_duration;
      $self->{'_topstart'}->{$nick} = $t_start;
   }
   return $change;
}

sub top_ranking {

   $self = shift;
   $nick = shift;
   $curtime = time();
   *DATA = $self->{'_userlist'};
   my $change = 0;
   foreach $chanuser (sort { $DATA{$a} <=> $DATA{$b} } keys %DATA) {
      my ($s_chan,$s_user) = split($self->{'_delimiter'},$chanuser);
      my $s_start = $DATA{$chanuser};
      $change = $change + $self->top_check($s_user,$s_start,$curtime);
   }
   if ($change > 0) { $self->top_save(); }

   *DATA = $self->{'_topduration'};
   my $count = 1;
   foreach $user (sort { $DATA{$b} <=> $DATA{$a} } keys %DATA) {
      my $idle = $self->format_time($DATA{$user});
      $self->{'_connection'}->privmsg($nick, $count.". ".$user." ".$idle);
      if ($count >= $self->{'_config'}->get('IDLELISTSIZE')) { return; }
      $count++;
   }
}

sub top_nick {

   $self = shift;
   my ($nick,$search) = @_;
   *DATA = $self->{'_topduration'};
   if (defined($DATA{$search})) {
      my $idle = $self->format_time($DATA{$search});
      $self->{'_connection'}->privmsg($nick, $search." ".$idle);
   } else {
      $self->{'_connection'}->privmsg($nick, "Unknown nick '$search'.");
   }
}

sub help {
   $self = shift;
   my $nick = shift;
   if (scalar(@_) > 0) {
      $cmd = lc(shift);
   }
   if (length($cmd) > 0) {
      if ($cmd eq "idle") {
         $self->{'_connection'}->privmsg($nick,"idle           - Shows a list of the current top idle nicks.");
      }
      if ($cmd eq "idletop") {
         $self->{'_connection'}->privmsg($nick,"idletop [nick] - Shows a list/entry of the all top idle nicks.");
      }
   } else {
      # print list
      $self->{'_connection'}->privmsg($nick,"idle            - Idle nicks.");
      $self->{'_connection'}->privmsg($nick,"idletop         - Top idle nicks.");
   }
}
