#!/usr/local/bin/perl
#

use strict;
use Net::IRC;
use NoK::config;
use NoK::event;
use NoK::modules;

#
#  Load configuration
#
my $config = new NoK::config($ARGV[0]);

#
#  Create the IRC and Connection objects
#
my $irc = new Net::IRC;
print "Creating connection to IRC server...";
my $conn = $irc->newconn(Server   => ($config->get('IRCSERVER')  ||  'irc.stealth.net'),
			 Port     => ($config->get('IRCPORT') || 6667),
			 Nick     => $config->get('IRCNICK'),
			 Ircname  => $config->get('IRCNAME'))
    or die "irctest: Can't connect to IRC server.\n";
print " done.\n";


#
#  Here's some stuff to print at odd moments.
#
sub irc_error {
   my ($self, $nick, $text) = @_;
   $self->privmsg($nick,"Error: $text");
}

my $eventhandler = new NoK::event;

#
#  Here are the handler subroutines. Fascinating, huh?
#

# MOTD (Motto of the day) start.
sub on_motd_start {
   my $self = shift;
   $eventhandler->action("motd_start");
}

# MOTD (Motto of the day).
sub on_motd {
   my ($self, $event) = @_;
   $eventhandler->action("motd",$event->args);
}

# What to do when the bot successfully connects.
sub on_motd_end {
   my $self = shift;
   $eventhandler->action("connect");
   if (length($config->get('IRCPASSWORD')) > 0) {
      print "Login... ";
      $conn->privmsg("NickServ","IDENTIFY ".$config->get('IRCPASSWORD'));
      print " done.\n";
   }
   foreach my $channel (split(/;/,$config->get('IRCCHANNEL'))) {
      $self->join($channel);
   }
   $eventhandler->action("motd_end");
}

# Handles infos from irc-server.
sub on_info {
   my ($self, $event) = @_;
print "INFO$: ".($event->args)[1]."\n";
}

# Handles some messages you get when you connect
sub on_init {
    my ($self, $event) = @_;
    my (@args) = ($event->args);
    shift @args;
    my $text = join(" ",(@args));
    $eventhandler->action("init", $text);
}

# What to do when someone leaves a channel the bot is on.
sub on_part {
    my ($self, $event) = @_;
    my ($channel) = ($event->to)[0];
    $eventhandler->action("part", $channel, $event->nick, $event->userhost);
}

# What to do when someone joins a channel the bot is on.
sub on_join {
    my ($self, $event) = @_;
    $eventhandler->action("join", ($event->to)[0], $event->nick, $event->userhost);
}

# What to do when we receive a private PRIVMSG.
sub on_msg {
   my ($self, $event) = @_;
   my $ignore_list = ";".$config->get('IGNORE').";";
   if (index($ignore_list,";".$event->nick.";") == -1) {
      $eventhandler->action("msg", $event->nick, $event->userhost, $event->args);
   }
}

# What to do when we receive channel text.
sub on_public {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("public", ($event->to)[0], $event->nick, $event->userhost, $event->args);
   }
}

# What to do when we receive a message via DCC CHAT.
sub on_chat {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("chat", ($event->to)[0], $event->nick, $event->userhost, $event->args);
    }
}

# Prints the names of people in a channel when we enter.
sub on_names {
    my ($self, $event) = @_;
    my ($mynick, $sign, $channel, @list) = ($event->args);
    $eventhandler->action("names", $channel, @list);
}

# What to do when we receive a DCC SEND or CHAT request.
sub on_dcc {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("dcc", $event->nick, $event->userhost, $event->args);
    }
}

# What to do when we receive a DCC SEND or CHAT request.
sub on_dcc_open {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("dcc_open", $event->nick, $event->userhost, $event->args);
    }
}

# What to do when we receive a DCC SEND or CHAT request.
sub on_dcc_update {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("dcc_update", $event->nick, $event->userhost, $event->args);
    }
}

# What to do when we receive a DCC SEND or CHAT request.
sub on_dcc_close {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("dcc_close", $event->nick, $event->userhost, $event->args);
    }
}

# Yells about incoming CTCP PINGs.
sub on_ping {
    my ($self, $event) = @_;
    $eventhandler->action("ping", $event->nick, $event->userhost, $event->args);
    $self->ctcp_reply($event->nick, join (' ', ($event->args)));
}

# Gives lag results for outgoing PINGs.
sub on_ping_reply {
    my ($self, $event) = @_;
    my ($args) = ($event->args)[1];
    $args = time - $args;
    $eventhandler->action("ping_reply", $event->nick, $event->userhost, $args);
}

# Change our nick if someone stole it.
sub on_nick_taken {
    my ($self) = shift;
    $eventhandler->action("nick_taken");
    $self->nick(substr($self->nick, -1) . substr($self->nick, 0, 8));
}

# Display formatted CTCP ACTIONs.
sub on_action {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("action", ($event->to)[0], $event->nick, $event->userhost, $event->args);
    }
}

# Reconnect to the server when we die.
sub on_disconnect {
   my ($self, $event) = @_;
   $eventhandler->action("disconnect");
   print "Disconnected from ", $event->from(), " (",
	($event->args())[0], ")";
   if (length($self->server) > 0) {
      $self->connect();
      print " Attempting to reconnect...\n";
   } else {
      print "\n";
      $irc = undef;
      exit(0);
   }
}

# Version request.
sub on_version {
    my ($self, $event) = @_;
    $self->ctcp_reply($event->nick,"VERSION ".$config->get('PRGVERSION'));
    $eventhandler->action("version", $event->nick, $event->userhost);
}

# Look at the topic for a channel you join.
sub on_topic {
   my ($self, $event) = @_;
   my @args = $event->args();
   if ($event->type() eq 'topic' and $event->to()) {
      if ($event->to() ne "") {
         $eventhandler->action("topic", $event->to(), $args[0]);
      }
   } else {
      if ($args[1] ne "") {
         $eventhandler->action("topic", $args[1], $args[2]);
      }
   }
}

# Some mode changes
sub on_mode {
    my ($self, $event) = @_;
    $eventhandler->action("mode", ($event->to)[0], $event->nick, $event->userhost, $event->args);
    my ($mode, $user) = ($event->args);
    if ($mode eq "+o") {
	$eventhandler->action("op", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "-o") {
        $eventhandler->action("deop", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "+b") {
	$eventhandler->action("ban", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "-b") {
	$eventhandler->action("deban", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "+v") {
	$eventhandler->action("voice", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "-v") {
	$eventhandler->action("devoice", ($event->to)[0], $event->nick,
	    $event->userhost, $user);
    } elsif ($mode eq "+s") {
        $eventhandler->action("secret", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-s") {
        $eventhandler->action("desecret", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+p") {
        $eventhandler->action("privat", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-p") {
        $eventhandler->action("deprivat", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+m") {
        $eventhandler->action("moderate", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-m") {
        $eventhandler->action("demoderate", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+i") {
        $eventhandler->action("cinvite", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-i") {
        $eventhandler->action("decinvite", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+n") {
        $eventhandler->action("noextern", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-n") {
        $eventhandler->action("denoextern", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+t") {
        $eventhandler->action("opertopic", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "-t") {
        $eventhandler->action("deopertopic", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+l") {
        $eventhandler->action("maxnick", ($event->to)[0], $event->nick,
            $event->userhost, $user);
    } elsif ($mode eq "-l") {
        $eventhandler->action("demaxnick", ($event->to)[0], $event->nick,
            $event->userhost);
    } elsif ($mode eq "+k") {
        $eventhandler->action("chanpass", ($event->to)[0], $event->nick,
            $event->userhost, $user);
    } elsif ($mode eq "-k") {
        $eventhandler->action("dechanpass", ($event->to)[0], $event->nick,
            $event->userhost, $user);
    } else {
	print (" Unknown mode: $mode (Arg: $user)\n");
    }
}

sub on_kick {
    my ($self, $event) = @_;
    my ($channel, $reason) = ($event->args);
    $eventhandler->action("kick", $channel, $event->nick, $event->userhost, ($event->to)[0], $reason);
}

sub on_nick {
    my ($self, $event) = @_;
    $eventhandler->action("nick", $event->nick, $event->userhost, ($event->args)[0]);
}

sub on_quit {
    my ($self, $event) = @_;
    $eventhandler->action("quit", $event->nick, $event->userhost, ($event->args));
}

sub on_notice {
    my ($self, $event) = @_;
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
       $eventhandler->action("notice", $event->nick, $event->userhost, ($event->args));
    }
}

sub on_other {
    my ($self, $event) = @_;
    $event->dump;
}

sub on_invite {
    my ($self, $event) = @_;
    my ($channel) = ($event->args);
    my $ignore_list = ";".$config->get('IGNORE').";";
    if (index($ignore_list,";".$event->nick.";") == -1) {
        $eventhandler->action("invite", $channel, $event->nick, $event->userhost);
    }
}

sub on_error {
    my ($self, $event) = @_;
    $event->dump;
}

sub on_source {
    my ($self, $event) = @_;
    $self->time($event->nick);
    $self->ctcp_reply($event->nick,"SOURCE ".$config->get('PRGSOURCE'));
    $eventhandler->action("source", $event->nick, $event->userhost);
}

sub on_time {
    my ($self, $event) = @_;
    $event->dump;
}

sub on_clientinfo {
    my ($self, $event) = @_;
    $self->ctcp_reply($event->nick,"CLIENTINFO ".$config->get('PRGINFO'));
    $eventhandler->action("clientinfo", $event->nick, $event->userhost);
}

sub on_userinfo {
    my ($self, $event) = @_;
    $self->ctcp_reply($event->nick,"USERINFO ".$config->get('USERINFO'));
    $eventhandler->action("userinfo", $event->nick, $event->userhost);
}

sub on_errmsg {
    my ($self, $event) = @_;
    $event->dump;
}

sub on_finger {
    my ($self, $event) = @_;
    $self->ctcp_reply($event->nick,"FINGER ".$config->get('FINGER'));
    $eventhandler->action("finger", $event->nick, $event->userhost);
}

# check privileges
die "Bad user id ($<)" if $< != getpwnam('jabba');
die "Bad group id ($()" if $( != getgrnam('jabba');

print "Installing handler routines...";
$conn->add_handler('cping',\&on_ping);
$conn->add_handler('crping',\&on_ping_reply);
$conn->add_global_handler('msg',\&on_msg,2);
$conn->add_handler('chat',\&on_chat);
$conn->add_handler('public',\&on_public);
$conn->add_handler('caction',\&on_action);
$conn->add_handler('join',\&on_join);
$conn->add_handler('part',\&on_part);
$conn->add_handler('cdcc',\&on_dcc);
$conn->add_handler('dcc_open',\&on_dcc_open);
$conn->add_handler('dcc_update',\&on_dcc_update);
$conn->add_handler('dcc_close',\&on_dcc_close);
$conn->add_handler('topic',\&on_topic);
#$conn->add_handler('notopic',\&on_topic);
$conn->add_handler('cversion',\&on_version);
$conn->add_handler('mode',\&on_mode);
$conn->add_handler('kick',\&on_kick);
$conn->add_handler('nick',\&on_nick);
$conn->add_handler('quit',\&on_quit);
$conn->add_handler('notice',\&on_notice);
$conn->add_handler('invite',\&on_invite);
$conn->add_handler('other',\&on_other);
$conn->add_handler('error',\&on_error);
$conn->add_handler('cclientinfo',\&on_clientinfo);
$conn->add_handler('csource',\&on_source);
$conn->add_handler('cuserinfo',\&on_userinfo);
$conn->add_handler('crtime',\&on_time);

$conn->add_global_handler([ 251,252,253,254,302,255 ], \&on_init);
$conn->add_global_handler('disconnect', \&on_disconnect);
$conn->add_global_handler(371, \&on_info);
$conn->add_global_handler(372, \&on_motd);
$conn->add_global_handler(375, \&on_motd_start);
$conn->add_global_handler(376, \&on_motd_end);
$conn->add_global_handler(433, \&on_nick_taken);
$conn->add_global_handler(353, \&on_names);
print " done.\n";

#$conn->debug(1);

# load module
my $module = new NoK::modules ($conn, $config, $eventhandler);

print "starting bot...\n";
$irc->start;
